// @ts-check
const fs = require("fs");
const net = require("net");
const path = require("path");
const _ = require("underscore");
const seq = require("promise-sequential");
const EventEmitter = require("events").EventEmitter;
const Constants = require("../constants");
const { deserialize } = require("../util/array");
const { messageReader } = require("./communication/MessageReader");
const { funktionsLib } = require("./communication/FunktionsLib");
const { packetCreator } = require("./communication/PacketCreator");

// Globals
const log = console.log;
const OK_PACKED = true;
const WRITE_4BYTE = 0x99;
const STOP_EVENT = "STOP";
const LEVEL_DATA = ["GainReduktionRMS", "OutputLevel", "InputLevel"];
const IGNOREABLE_WRITE_MESSAGES = ["storeLiveParameters", "setAutoSendUI", "setSendLevelData", "requestOkMessage"];

class Device {
    /**
     * @param {string} address
     * @param {number} port
     * @param {number | any} id
     */
    constructor(networkManager, address, port, id = null) {
        // Init
        this.type = null;
        this.error = null;
        this.port = port;
        this.writeFlag = 0;
        this.properties = {};
        this.address = address;
        this.events = new EventEmitter();
        this.networkManager = networkManager;
        this.id = id || parseInt(address.split(".").pop());
        this.log = {
            in: [],
            out: []
        };
    }

    start() {
        log(`Device #${this.id}: Connected`);

        // Create client and connect
        this.client = net.createConnection(
            {
                host: this.address,
                port: this.port
            },
            () => {
                console.log("Connected");
            }
        );

        // Configure client
        this.client.on("data", data => this.read(data));

        // Configure client
        this.client.on("error", data => {
            log("ERROR", data);
            this.networkManager.emit("deviceerror", {
                moduleId: this.id,
                error: data
            });

            this.stop();
        });
    }

    stop() {
        log(`Device #${this.id}: Disconnected`);

        // Set destroyed flag
        this.destroyed = true;

        // Fire local event
        this.events.emit("message", STOP_EVENT);

        // Destroy client
        if (this.client) {
            this.client.destroy();
        }
    }

    /**
     * @param {Buffer} data
     */
    read(data) {
        var messages = messageReader.read([...data], this);
        _.each(messages, message => {
            if (message.id == "Heartbeat" || message.id == "HardwareAmpStatus") {
                // Ignore
            } else if (_.contains(LEVEL_DATA, message.id)) {
                // Ignore
            } else {
                // Initialization process
                // if (_.contains(["RMSLimit", "PeakLimit", "ActualPreset"], message.id)) {
                if (message.id == "ActualPreset") {
                    // Add actual preset as prop
                    this.properties.ActualPreset = message.value;

                    // Handle counters
                    if (this.counters) {
                        this.counters[message.id].inc();
                        const next = !_.find(this.counters, counter => !counter.done());
                        if (next) {
                            this.write("readDeviceName", null);
                            this.counters = null;
                        }
                    }

                    // Limiter
                    if (message.id != "ActualPreset") {
                        this.networkManager.response(this.id, message);
                    }
                } else if (message.id == "DeviceName" && message.value && !this.type) {
                    this.type = message.value.trim();
                    this.properties.ActualSpeakerPresets = this.properties.ActualSpeakerPresets || {};
                    this.executeFile(path.join(__dirname, "../data/init/readDevices.json")).then(() => {
                        this.executeFile(path.join(__dirname, "../data/init/readMyData.json")).then(() => {
                            const q = _.map(
                                this.properties.ActualSpeaker,
                                (slotId, channelId) => () =>
                                    this.write("readSpeakerName", {
                                        SlotNr: this.properties.ActualSpeaker[channelId]
                                    }).then(
                                        res => {
                                            this.properties.ActualSpeakerPresets[channelId] = res.value;
                                        },
                                        e => {
                                            this.properties.ActualSpeakerPresets[channelId] = "ERROR";
                                        }
                                    )
                            );

                            // When finished, discover device
                            seq(q);
                        });
                    });
                } else if (message.id.indexOf("SpeakerName") === 0 || message.id.indexOf("PresetName") === 0 || _.contains(["FirmwareOEM_id", "FW_OEMSerial", "FW_Seriennummer", "FW_VERSION", "FW_DATUM", "FW_VERSION", "UserBank1", "UserBank2", "SpeakerName", "SpeakerLibName", "UserName", "PresetLibName", "ActualPreset", "ActualSpeaker", "AutoTest", "AmpStateMeta"], message.id)) {
                    const value = _.isString(message.value) ? message.value.trim() : message.value;
                    if (message.id.indexOf("SpeakerName") === 0) {
                        this.properties.SpeakerName = this.properties.SpeakerName || {};
                        this.properties.SpeakerName[message.index] = value;
                    } else if (message.id.indexOf("PresetName") === 0) {
                        this.properties.PresetName = this.properties.PresetName || {};
                        this.properties.PresetName[message.index] = value;
                    } else if (message.id.indexOf("AmpStateMeta") === 0) {
                        this.properties.AmpStateMeta = deserialize(value);
                    } else if (message.channelId >= 0) {
                        this.properties[message.id] = this.properties[message.id] || {};
                        this.properties[message.id][message.channelId] = value;
                    } else {
                        this.properties[message.id] = value;
                    }

                    // FIX für falschen "device name" im Gerät (vom Werk)
                    if (message.id == "FW_OEMSerial" && value > 0) {
                        const valueAsString = "" + value;
                        if (valueAsString.length > 3) {
                            const deviceType = Constants.AMP_SN_NR_ID[valueAsString[3]];
                            if (deviceType && this.type != deviceType) {
                                this.type = deviceType;
                                this.write("setDeviceName", {
                                    name: deviceType
                                });
                            }
                        }
                    }
                }

                console.log(this.id, message);
                this.events.emit("message", message);
                this.networkManager.response(this.id, message);
            }
        });
    }

    /**
     * @param {string} commandId
     * @param {*} params
     */
    write(commandId, params) {
        if (this.destroyed) {
            return Promise.reject(new Error(`Device ${this.id} disconnected`));
        }

        const message = _.isArray(commandId) ? _.compact(_.map(_.compact(commandId), cmd => this.build(cmd.commandId || cmd.id || cmd, cmd.params))) : this.build(commandId, params);
        if (message && (message.data || message.length > 0)) {
            console.log(this.id, commandId, params);
            return this.execute(message);
        }

        return Promise.resolve();
    }

    /**
     * Build messag from command id and parameters
     * @param {string} commandId
     * @param {any} params
     */
    build(commandId, params) {
        const fn = funktionsLib[commandId];
        if (_.isFunction(fn)) {
            try {
                const msg = fn.call(funktionsLib, params);
                if (this.isWriteMessage(msg) && !_.contains(IGNOREABLE_WRITE_MESSAGES, commandId)) {
                    this.writeFlag = new Date().getTime();
                }

                return msg;
            } catch (e) {
                console.warn(e);
            }
        }

        return null;
    }

    /**
     * @param {*} message
     */
    execute(message) {
        return new Promise((resolve, reject) => {
            // Is client?
            if (!this.client) {
                return reject(new Error("Device not connected"));
            }

            // Init
            let timeout;

            // Build desired response
            const responsePattern = message.response || (OK_PACKED && _.isArray(message) && _.size(message) > 0 && _.last(message).response);

            // Listens for other messages during callback phase
            const listener = incoming => {
                // Check if incoming messages matches the given response pattern
                if (incoming == STOP_EVENT || _.isMatch(incoming, responsePattern)) {
                    // Yeah! Remove listener, clear timeout and resolve the promise
                    this.events.removeListener("message", listener);
                    clearTimeout(timeout);

                    // Check for errors
                    if (incoming == STOP_EVENT || incoming.error) {
                        reject(new Error(incoming == STOP_EVENT ? STOP_EVENT : incoming.error));
                    } else {
                        resolve(incoming);
                    }
                }
            };

            // If we expect an response, register listener
            if (responsePattern) {
                this.events.on("message", listener);
            }

            // Actually write data to device
            try {
                let packet;
                if (message.packet) {
                    packet = message.packet;
                } else {
                    const messageData = _.isArray(message) ? _.compact(_.pluck(_.flatten(message), "data")) : message.data;
                    packet = Buffer.from(packetCreator.makePacket(messageData));
                }

                this.client.write(packet, err => {
                    // Handle sending error
                    if (err) {
                        // If we've registered a listener, remove it
                        if (responsePattern) {
                            this.events.removeListener("message", listener);
                        }

                        // Finally reject the promise
                        return reject(err);
                    }

                    if (!responsePattern) {
                        // If there is no response needed, resolve promise
                        resolve(false);
                    } else {
                        // If there is no response found, reject after x ms
                        timeout = setTimeout(() => {
                            this.events.removeListener("message", listener);
                            reject(new Error("Timeout: " + JSON.stringify(responsePattern)));
                        }, message.timeout || 5000);
                    }
                });
            } catch (err) {
                // If we've registered a listener, remove it
                if (responsePattern) {
                    this.events.removeListener("message", listener);
                }

                // Finally reject the promise
                reject(err);
            }
        });
    }

    executeFile(filename) {
        return this.loadFile(filename).then(script => {
            const q = _.map(
                _.filter(script.packets, packet => {
                    if (_.isArray(packet.deviceTypes)) {
                        return _.contains(packet.deviceTypes, this.type);
                    } else if (_.isString(packet.deviceTypes)) {
                        return this.type == packet.deviceTypes;
                    }

                    return true;
                }),
                packet => {
                    return () => {
                        // Build message
                        const message = {
                            packet: Buffer.from(packet.data, "base64"),
                            response: packet.response,
                            timeout: packet.timeout
                        };

                        // Log
                        console.log(this.id, message);

                        // Execute message
                        return this.execute(message);
                    };
                }
            );

            return seq(q);
        });
    }

    loadFile(filename) {
        return new Promise((resolve, reject) => fs.readFile(filename, (err, data) => (err ? reject(err) : resolve(JSON.parse(data.toString())))));
    }

    isWriteMessage(msg) {
        if (_.isArray(msg)) {
            return _.contains(
                _.map(msg, item => this.isWriteMessage(item)),
                true
            );
        }

        return msg && msg.data && msg.data.data && msg.data.data.length > 0 && msg.data.data[0] === WRITE_4BYTE;
    }
}

module.exports = Device;
